<?php
session_start();
require '../../../script/conexion.php';

ini_set('display_errors', 0);
error_reporting(E_ALL);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Método no permitido');
}

function post($k, $default = '') {
    return trim($_POST[$k] ?? $default);
}

/* ---------------- Entradas ---------------- */
$fecha = date('Y-m-d');

$nombre_usuario   = post('nombre_usuario');
$nombre_categoria = post('nombre_categoria');
$nombre_maquina   = post('nombre_maquina');
$color            = post('color');
$color2           = post('color2');

/* ---------------- Validaciones ---------------- */
$errores = [];
if ($nombre_usuario === '')   $errores[] = 'Falta nombre_usuario';
if ($nombre_categoria === '') $errores[] = 'Falta nombre_categoria';
if ($nombre_maquina === '')   $errores[] = 'Falta nombre_maquina';
if ($color === '')            $errores[] = 'Falta color';
if ($color2 === '')           $errores[] = 'Falta color2';

if ($errores) {
    http_response_code(400);
    exit('Error: ' . implode(', ', $errores));
}

try {
    $conn->begin_transaction();

    /* -------------------------------------------------------
       1) Traer datos de máquina
    ------------------------------------------------------- */
    $stmtM = $conn->prepare("SELECT puestos, rotativa, canones, limite_puestos, duo FROM maquina WHERE nombre = ? LIMIT 1");
    if (!$stmtM) throw new Exception("Prepare máquina: " . $conn->error);

    $stmtM->bind_param("s", $nombre_maquina);
    $stmtM->execute();
    $resM = $stmtM->get_result();
    $row8 = $resM->fetch_assoc();
    $stmtM->close();

    if (!$row8) {
        throw new Exception("La máquina no existe");
    }

    $puestos        = $row8['puestos'];
    $rotativa       = $row8['rotativa'];
    $canones        = $row8['canones'];
    $limite_puestos = $row8['limite_puestos'];
    $duo            = $row8['duo'];

    $hora = '';

    /* -------------------------------------------------------
       2) Generar consecutivo OT de forma segura (lock)
          - esto evita que 2 personas creen el mismo numero
    ------------------------------------------------------- */
    // Ojo: si numero_orden_trabajo es VARCHAR, igual sirve para extraer dígitos.
    $stmtMax = $conn->prepare("SELECT MAX(numero_orden_trabajo) AS mayor FROM orden_trabajo FOR UPDATE");
    if (!$stmtMax) throw new Exception("Prepare max OT: " . $conn->error);

    $stmtMax->execute();
    $resMax = $stmtMax->get_result();
    $rowMax = $resMax->fetch_assoc();
    $stmtMax->close();

    $ultimo = $rowMax['mayor'] ?? '';
    $numero = $ultimo ? (int)preg_replace('/[^0-9]+/', '', $ultimo) : 0;
    $codigo = (string)($numero + 1);

    /* -------------------------------------------------------
       3) Insertar ORDEN DE TRABAJO (prepared)
    ------------------------------------------------------- */
    $stmtOT = $conn->prepare("
        INSERT INTO orden_trabajo
        (fecha, numero_orden_trabajo, nombre_usuario, nombre_empleado, nombre_maquina,
         puestos, rotativa, canones, limite_puestos, duo, hora_desde, hora_hasta, observacion)
        VALUES
        (?,?,?,?,?,?,?,?,?,?,?,?,?)
    ");
    if (!$stmtOT) throw new Exception("Prepare insert OT: " . $conn->error);

    $nombre_empleado = null;
    $observacion_ot  = 'Sin comentarios';

    $stmtOT->bind_param(
        "sssssssssssss",
        $fecha,
        $codigo,
        $nombre_usuario,
        $nombre_empleado,
        $nombre_maquina,
        $puestos,
        $rotativa,
        $canones,
        $limite_puestos,
        $duo,
        $hora,
        $hora,
        $observacion_ot
    );

    if (!$stmtOT->execute()) throw new Exception("Insert OT falló: " . $stmtOT->error);
    $id_orden_trabajo = (int)$conn->insert_id;
    $stmtOT->close();

    if ($id_orden_trabajo <= 0) {
        throw new Exception("No se pudo obtener id_orden_trabajo");
    }

    /* -------------------------------------------------------
       4) Traer referencias a producir (por categoría + color)
          (equivalente a tu query con GROUP BY)
    ------------------------------------------------------- */
    $stmtRefs = $conn->prepare("
        SELECT rp.referencia_producto, p.referencia_segmento
        FROM referencia_produccion rp
        JOIN producto p ON p.referencia = rp.referencia_producto
        WHERE p.categoria = ? AND p.color = ? AND p.color2 = ?
        GROUP BY rp.referencia_producto, p.referencia_segmento
        ORDER BY rp.referencia_producto ASC
    ");
    if (!$stmtRefs) throw new Exception("Prepare referencias: " . $conn->error);

    $stmtRefs->bind_param("sss", $nombre_categoria, $color, $color2);
    $stmtRefs->execute();
    $resRefs = $stmtRefs->get_result();

    /* Preparar statements reutilizables */
    $stmtTallas = $conn->prepare("
        SELECT rp.talla
        FROM referencia_produccion rp
        JOIN producto p ON p.referencia = rp.referencia_producto
        WHERE rp.referencia_producto = ?
        GROUP BY rp.talla
        ORDER BY rp.talla ASC
    ");
    if (!$stmtTallas) throw new Exception("Prepare tallas: " . $conn->error);

    $stmtSum = $conn->prepare("
        SELECT COALESCE(SUM(rp.cantidad),0) AS cantidad
        FROM referencia_produccion rp
        JOIN producto p ON p.referencia = rp.referencia_producto
        WHERE rp.referencia_producto = ? AND rp.talla = ?
    ");
    if (!$stmtSum) throw new Exception("Prepare sum cantidad: " . $conn->error);

    $stmtMoldeCount = $conn->prepare("
        SELECT COUNT(id) AS contar_molde_combinado
        FROM molde_combinado
        WHERE referencia_segmento = ? AND (talla_1 = ? OR talla_2 = ?)
    ");
    if (!$stmtMoldeCount) throw new Exception("Prepare molde count: " . $conn->error);

    $stmtInsDet = $conn->prepare("
        INSERT INTO orden_trabajo_detalle
        (id_orden_trabajo, referencia_segmento, referencia_producto, talla, cantidad, combinado, contar, orden)
        VALUES (?,?,?,?,?,?,?,?)
    ");
    if (!$stmtInsDet) throw new Exception("Prepare insert detalle OT: " . $conn->error);

    $item = 1;

    while ($row1 = $resRefs->fetch_assoc()) {
        $referencia = $row1['referencia_producto'];
        $referencia_segmento = $row1['referencia_segmento'];
        $orden = $item;

        // tallas por referencia
        $stmtTallas->bind_param("s", $referencia);
        $stmtTallas->execute();
        $resTallas = $stmtTallas->get_result();

        while ($row2 = $resTallas->fetch_assoc()) {
            $talla = $row2['talla'];

            // sumar cantidad por ref + talla
            $stmtSum->bind_param("ss", $referencia, $talla);
            $stmtSum->execute();
            $resSum = $stmtSum->get_result();
            $row3 = $resSum->fetch_assoc();
            $cantidad = (int)($row3['cantidad'] ?? 0);

            // validar molde combinado
            $stmtMoldeCount->bind_param("sss", $referencia_segmento, $talla, $talla);
            $stmtMoldeCount->execute();
            $resMC = $stmtMoldeCount->get_result();
            $row5m = $resMC->fetch_assoc();
            $combinado = ((int)($row5m['contar_molde_combinado'] ?? 0)) > 0 ? 'Si' : 'No';

            // Insert detalle OT
            $contar = 'Si'; // igual a tu lógica inicial
            $stmtInsDet->bind_param(
                "isssisss",
                $id_orden_trabajo,
                $referencia_segmento,
                $referencia,
                $talla,
                $cantidad,
                $combinado,
                $contar,
                $orden
            );

            if (!$stmtInsDet->execute()) {
                throw new Exception("Insert OT detalle falló: " . $stmtInsDet->error);
            }
        }

        $item++;
    }

    $stmtRefs->close();
    $stmtTallas->close();
    $stmtSum->close();
    $stmtMoldeCount->close();
    $stmtInsDet->close();

    /* -------------------------------------------------------
       5) Ajustar "contar" para moldes combinados
          (misma lógica que tienes)
    ------------------------------------------------------- */
    $stmtCombi = $conn->prepare("
        SELECT id, referencia_segmento, referencia_producto, talla, cantidad
        FROM orden_trabajo_detalle
        WHERE id_orden_trabajo = ? AND combinado = 'Si'
    ");
    if (!$stmtCombi) throw new Exception("Prepare select combinados: " . $conn->error);

    $stmtCombi->bind_param("i", $id_orden_trabajo);
    $stmtCombi->execute();
    $resCombi = $stmtCombi->get_result();

    $stmtMoldeGet = $conn->prepare("
        SELECT talla_1, talla_2
        FROM molde_combinado
        WHERE referencia_segmento = ? AND (talla_1 = ? OR talla_2 = ?)
        LIMIT 1
    ");
    if (!$stmtMoldeGet) throw new Exception("Prepare molde get: " . $conn->error);

    $stmtSumT = $conn->prepare("
        SELECT COALESCE(SUM(rp.cantidad),0) AS cantidad
        FROM referencia_produccion rp
        JOIN producto p ON p.referencia = rp.referencia_producto
        WHERE rp.referencia_producto = ? AND rp.talla = ?
    ");
    if (!$stmtSumT) throw new Exception("Prepare sumT: " . $conn->error);

    $stmtUpdContar = $conn->prepare("
        UPDATE orden_trabajo_detalle
        SET contar = ?
        WHERE id = ? AND id_orden_trabajo = ?
        LIMIT 1
    ");
    if (!$stmtUpdContar) throw new Exception("Prepare update contar: " . $conn->error);

    while ($row10 = $resCombi->fetch_assoc()) {
        $detalle_id      = (int)$row10['id'];
        $segmento_combi  = $row10['referencia_segmento'];
        $referencia_combi= $row10['referencia_producto'];
        $talla_combi     = $row10['talla'];

        // traer talla_1 y talla_2 del molde
        $stmtMoldeGet->bind_param("sss", $segmento_combi, $talla_combi, $talla_combi);
        $stmtMoldeGet->execute();
        $resMG = $stmtMoldeGet->get_result();
        $molde = $resMG->fetch_assoc();

        if (!$molde) {
            // si no existe molde, por seguridad dejamos contar=Si
            $contar = 'Si';
        } else {
            $talla_1 = $molde['talla_1'];
            $talla_2 = $molde['talla_2'];

            // cantidades por cada talla combinada
            $stmtSumT->bind_param("ss", $referencia_combi, $talla_1);
            $stmtSumT->execute();
            $c1 = (int)($stmtSumT->get_result()->fetch_assoc()['cantidad'] ?? 0);

            $stmtSumT->bind_param("ss", $referencia_combi, $talla_2);
            $stmtSumT->execute();
            $c2 = (int)($stmtSumT->get_result()->fetch_assoc()['cantidad'] ?? 0);

            // tu misma lógica
            if ($c1 > $c2) {
                $contar = ($talla_combi == $talla_1) ? 'Si' : 'No';
            } elseif ($c2 > $c1) {
                $contar = ($talla_combi == $talla_2) ? 'Si' : 'No';
            } else {
                $contar = ($talla_combi == $talla_1) ? 'Si' : 'No';
            }
        }

        $stmtUpdContar->bind_param("sii", $contar, $detalle_id, $id_orden_trabajo);
        if (!$stmtUpdContar->execute()) {
            throw new Exception("Update contar falló: " . $stmtUpdContar->error);
        }
    }

    $stmtCombi->close();
    $stmtMoldeGet->close();
    $stmtSumT->close();
    $stmtUpdContar->close();

    $conn->commit();

    header("Location: ../orden-trabajo-detalle.php?id_orden_trabajo=" . $id_orden_trabajo);
    exit;

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    exit("Error: " . $e->getMessage());
}
